;----------------------------------------------------------------------------------------------------------
; NAME:
;
;	OMTI_ROUTINES
;
; PURPOSE:
;
;	This routine visualizes OMTI all-sky imager data at Resolute Bay.
;
; AUTHOR:
;
;	Keisuke Hosokawa (The Univ. of Electro-Communications, Tokyo, Japan)
;
;
;----------------------------------------------------------------------------------------------------------
; NAME:
;
;	COMMON BLOCKS
;

	COMMON data_omti,	img_omti,keo_omti,zan_omti

	COMMON pref_omti,	num_omti,now_omti,flt_omti,col_omti,cnt_omti,avl_omti,alt_omti, $
				day_omti,hms_omti,hht_omti,mmt_omti,sst_omti,tsc_omti,jday_omti, $
				xra_omti,yra_omti,xce_omti,yce_omti,crd_omti,app_omti, $
				omti_keo_stt,omti_keo_end,lev_omti,omti_keo_stt_org,omti_keo_end_org, $
				glo_omti,gla_omti,mlo_omti,mla_omti,lat_rsb,lon_rsb, $
				omti_data_dir,omti_cord_dir
        
;----------------------------------------------------------------------------------------------------------
; NAME:
;
;	FIND_NEXT_PREV_DAY
;

        FUNCTION find_next_prev_day,old_day,next=next,prev=prev

; Next day or previous day?
        IF KEYWORD_SET(next) THEN offset=+1
        IF KEYWORD_SET(prev) THEN offset=-1

; Break original date into yy,mm,dd
        yy=FIX(STRMID(old_day,0,4))
	mm=FIX(STRMID(old_day,4,2))
	dd=FIX(STRMID(old_day,6,2))

; Find the date that you want
        jday_tommorrow=JULDAY(mm,dd,yy)+offset
        CALDAT,jday_tommorrow,mf,df,yf
        new_day=STRING(yf,FORMAT='(I4.4)')+STRING(mf,FORMAT='(I2.2)')+STRING(df,FORMAT='(I2.2)')

; Return it
        RETURN,new_day

        END

;----------------------------------------------------------------------------------------------------------
; NAME:
;
;	START_OMTI
;
; PURPOSE:
;
;	Initialize arrays and getting started omti_routines. Default is OI630 wavelength.
;
; EXAMPLE:
;
;	Go > start_omti
;

	PRO start_omti

; Red line (OI 630 nm) is default
	set_flt_omti,'red'
	set_col_omti,3

; Initialise arrays
	initialise_omti

	END

;----------------------------------------------------------------------------------------------------------
; NAME:
;
;	INITIALISE_OMTI
;
; PURPOSE:
;
;	Initialize arrays and get started.
;
; EXAMPLE:
;
;	Go > initialise_omti
;

	PRO initialise_omti

	COMMON data_omti
	COMMON pref_omti

; Set data directory
	omti_data_dir='/radar07/data/Imager/abs'
	omti_cord_dir='/radar07/data/Imager/map'

; Image range
	xra_omti=[28,250]
	yra_omti=[ 0,222]
	xce_omti=0.5*(xra_omti(0)+xra_omti(1))
	yce_omti=0.5*(yra_omti(0)+yra_omti(1))

; Data size
        size=256
	size_keo=xra_omti(1)-xra_omti(0)
        ncnt=1440
        num_omti=ncnt

; Setting arrays for the data itself
        img_omti=INTARR(size,size,ncnt)
        keo_omti=INTARR(4,size_keo,ncnt)
        avl_omti=INTARR(ncnt)
	lev_omti=[0,3000]

; Location info.
	gla_omti=FLTARR(size,size)
	glo_omti=FLTARR(size,size)
	mla_omti=FLTARR(size,size)
	mlo_omti=FLTARR(size,size)
	zan_omti=FLTARR(size,size)

; Time info.
	day_omti='not date specified'
        hms_omti=STRARR(ncnt)
        hht_omti=INTARR(ncnt)
        mmt_omti=INTARR(ncnt)
        sst_omti=INTARR(ncnt)
        tsc_omti=LONARR(ncnt)

; Initialize
	time_omti,0000,2400
	set_coords_omti,'mag'

; Latitude and longitude of the site
	lat_rsb=74.730
	lon_rsb=265.07

	END

;----------------------------------------------------------------------------------------------------------
; NAME:
;
;	GO_OMTI
;
; PURPOSE:
;
;	Jump to specific scan.
;
; EXAMPLE:
;
;	Go > go_omti,20
;

	PRO go_omti,scan

	COMMON pref_omti

	IF N_PARAMS() EQ 1 THEN BEGIN

		IF scan LT 0 AND scan GT 1439 THEN BEGIN
			PRINT,'ERROR: Scan must be in between 0 and 1439'
			RETURN
		ENDIF

		IF avl_omti(scan) EQ 1 THEN BEGIN
			now_omti=scan
		ENDIF ELSE BEGIN
			PRINT,'WARNING: Image out of range (i.e. you chose daytime!)'
			now_omti=1500
		ENDELSE
	ENDIF ELSE BEGIN
		FOR i=0,719 DO BEGIN
			IF avl_omti(i) EQ 1 THEN BEGIN
				now_omti=i
				GOTO,LOOP_OUT
			ENDIF
		ENDFOR
		LOOP_OUT:
	ENDELSE

	END

;----------------------------------------------------------------------------------------------------------
; NAME:
;
;       GO_TIME_OMTI
;
; PURPOSE:
;
;       Jump to specified time.
;
; EXAMPLE:
;
;	Go > go_time_omti,1400
;

        PRO go_time_omti,scan_time

	COMMON pref_omti

; Total time in minutes
        scan_tmin=60*FIX(scan_time/100)+scan_time-100*FIX(scan_time/100)

; Scan which we jump into
	scan_found=scan_tmin/2

; Jump there
       	go_omti,scan_found

        END

;----------------------------------------------------------------------------------------------------------
; NAME:
;
;       SET_COORDS_OMTI
;
; PURPOSE:
;
; 	Setting coordinate type.
;
; EXAMPLE:
;
;	For geomagnetic (AACGM)
;
;		Go > set_coords_omti,'mag'
;
;	For geographic
;
;		Go > set_coords_omti,'geog'
;

	PRO set_coords_omti,coord_name

	COMMON pref_omti

	IF coord_name NE 'geog' AND coord_name NE 'mag' THEN BEGIN
		PRINT,'WARNING: Invalid coordinate system!'
		RETURN
	ENDIF

	IF coord_name EQ 'geog' THEN crd_omti=0
	IF coord_name EQ 'mag'  THEN crd_omti=1

	END
	
;----------------------------------------------------------------------------------------------------------
; NAME:
;
;       SET_FLT_OMTI
;
; PURPOSE:
;
; 	Setting filter type.
;
; EXAMPLE:
;
;	For red line (OI 630.0 nm)
;
;		Go > set_flt_asi,'red'
;
;	For green line (OI 557.7 nm)
;
;		Go > set_flt_asi,'gre'
;
;	For sodium line (Na 589.3 nm)
;
;		Go > set_flt_asi,'sod'
;

	PRO set_flt_omti,flt_type

	COMMON pref_omti

	CASE flt_type OF
		'gre':	flt_omti='C61'
		'red':	flt_omti='C62'
		'sod':	flt_omti='C66'
		ELSE:	BEGIN
				PRINT,'ERROR: invalid data type'
				PRINT,'       select one of [gre,red,sod] please!'
				RETURN
			END
	ENDCASE

; Setting colour table
	IF flt_omti EQ 'C61' THEN set_col_omti,8
	IF flt_omti EQ 'C62' THEN set_col_omti,3
	IF flt_omti EQ 'C66' THEN set_col_omti,1
	
	END

;----------------------------------------------------------------------------------------------------------
; NAME:
;
;       SET_COL_OMTI
;
; PURPOSE:
;
; 	Setting colour table to be used.
;
; EXAMPLE:
;
;	Go > set_col_omti,3
;

	PRO set_col_omti,col_num

	COMMON pref_omti

	col_omti=col_num
	LOADCT,col_num,/SILENT

	END

;----------------------------------------------------------------------------------------------------------
; NAME:
;
;	SET_LEV_OMTI
;
; PURPOSE:
;
;	Setting minimum and maximum values to be used in colour plot.
;
; EXAMPLE:
;
;	Go > set_lev_omti,0,1000
;

	PRO set_lev_omti,min_val,max_val

	COMMON pref_omti

; Setup level
        lev_omti=[min_val,max_val]

	END

;----------------------------------------------------------------------------------------------------------
; NAME:
;
;       TIME_OMTI
;
; PURPOSE:
;
;       Set duration time for time-series plotting.
;
; EXAMPLE:
;
;       Go > time_omti,1200,1800
;

        PRO time_omti,time_a,time_b
        
	COMMON pref_omti

; Setting start and end time in hhmm
	omti_keo_stt_org=time_a
	omti_keo_end_org=time_b

; Setting start and end time in total sec
        omti_keo_stt=60L*60L*(time_a/100)+60L*(time_a-(time_a/100)*100)
        omti_keo_end=60L*60L*(time_b/100)+60L*(time_b-(time_b/100)*100)

        END

;----------------------------------------------------------------------------------------------------------
; NAME:
;
;	FILE_OMTI
;
; PURPOSE:
;
;	Routines for reading the raw and abs data.
;
; EXAMPLE:
;
;	Go > file_omti,'20050111'
;
; NOTE:
;
;	IF you want to make average and deviation files, add make_dev keyword.
;

	PRO file_omti,date_eight,append=append,no_print=no_print

	COMMON data_omti
	COMMON pref_omti

; Initialise variables
	IF NOT KEYWORD_SET(append) THEN initialise_omti

	IF day_omti EQ date_eight THEN BEGIN
		PRINT,'The data you specified were already loaded!'
		RETURN
	ENDIF

	IF KEYWORD_SET(append) THEN app_omti=1 ELSE app_omti=0

; Date identification
	day_omti=date_eight
	date=STRMID(date_eight,2,6)
        yy=FIX(STRMID(day_omti,0,4))
	mm=FIX(STRMID(day_omti,4,2))
        dd=FIX(STRMID(day_omti,6,2))
        jday_omti=JULDAY(mm,dd,yy)-JULDAY(1,0,yy)

; Setting arrays for temporal reading process
	size=256
	dummy=INTARR(4)
	cnt_tmp=INTARR(size,size)

; List up files
	PRINT,' '
	PRINT,'Now reading data ...'
	PRINT,' '
	data_head=omti_data_dir+'/'+date+'/'+flt_omti+'_'+date
	command='ls '+data_head+'* > ./fname.lst'
	SPAWN,command

; Open list file
	OPENR,lst_lun,'./fname.lst',/GET_LUN

; Loop for files begins
	WHILE NOT EOF(lst_lun) DO BEGIN

; Read one filename
		fname='resolute bay is really cold!'
		READF,lst_lun,fname

; Identify time
		hms_tmp=STRMID(fname,STRLEN(data_head),6)
		IF KEYWORD_SET(append) THEN time_offs=86400L ELSE time_offs=0L
		tsc_tmp=time_offs+3600L*FIX(STRMID(hms_tmp,0,2))+ $
			60L*FIX(STRMID(hms_tmp,2,2))+FIX(STRMID(hms_tmp,4,2))
		scans=tsc_tmp/120L
		IF NOT KEYWORD_SET(no_print) THEN $
			PRINT,STRING(scans,FORMAT='(I4.4)')+'th scan: '+fname
		hms_omti(scans)=STRMID(fname,STRLEN(data_head),6)
		hht_omti(scans)=FIX(STRMID(hms_tmp,0,2))
		mmt_omti(scans)=FIX(STRMID(hms_tmp,2,2))
		sst_omti(scans)=FIX(STRMID(hms_tmp,4,2))
		tsc_omti(scans)=tsc_tmp

; Check data avairability
		avl_omti(scans)=1

; Capture 2-D image
		OPENR,omti_lun,fname,/GET_LUN,/SWAP_ENDIAN
		READU,omti_lun,dummy,cnt_tmp
		FREE_LUN,omti_lun

; Convert the image if required
		BYTEORDER,cnt_tmp,/SSWAP
          	cnt_tmp(*,*)=ROTATE(cnt_tmp,7)
          	img_omti(*,*,scans)=FIX(cnt_tmp(*,*)*0.25)

; Capture the data into keogram
		FOR j=0,xra_omti(1)-xra_omti(0)-1 DO BEGIN
			keo_omti(0,j,scans)=img_omti(xra_omti(0)+j,j,scans)
			keo_omti(1,j,scans)=img_omti(xra_omti(1)-j,j,scans)
			keo_omti(2,j,scans)=img_omti(xce_omti,j,scans)
			keo_omti(3,j,scans)=img_omti(xra_omti(0)+j,yce_omti,scans)
		ENDFOR
	ENDWHILE

; Close list file and delete it
	FREE_LUN,lst_lun
	SPAWN,'rm ./fname.lst'

	dark_scans=WHERE(avl_omti EQ 1,no_dark_scans)

; Capture fov
	alt=0
	IF flt_omti EQ 'C61' THEN alt=110	
	IF flt_omti EQ 'C62' THEN alt=250
	file_omti_fov,alt=alt

; Initialize now_omti
	go_omti

; Print info.
	print_omti_info

	END

;----------------------------------------------------------------------------------------------------------
; NAME:
;
;	PRINT_OMTI_INFO
;

	PRO print_omti_info,image_info=image_info

	COMMON data_omti
	COMMON pref_omti

; Julian day info.
        jday_omti_str=STRING(jday_omti,FORMAT='(I3.3)')

; Filter info.
        IF flt_omti EQ 'C61' THEN flt_omti_str='C61 (557.7 nm)'
        IF flt_omti EQ 'C62' THEN flt_omti_str='C62 (630.0 nm)'
        IF flt_omti EQ 'C66' THEN flt_omti_str='C66 (589.3 nm)'

; Print info.
        PRINT,''
        PRINT,'---------------------------------------------------------------------------'
        PRINT,'Optical Mesosphere Thermosphere Imager (OMTI)                  Resolute Bay
        PRINT,'---------------------------------------------------------------------------'
        PRINT,'Date                                                         '+ $
		day_omti+' ('+jday_omti_str+')
        PRINT,'Filter Type                                                  '+flt_omti_str
        dark_scans=WHERE(avl_omti EQ 1,no_dark_scans)
        PRINT,'No. Available Images                                            '+ $
		STRCOMPRESS(STRING(no_dark_scans))+' images'
        PRINT,'Current Image                                                  '+ $
		STRING(now_omti,FORMAT='(I4.4)')+'th image'
        PRINT,'Current Image Time                                              '+ $
		STRMID(hms_omti(now_omti),0,4)+' '+STRMID(hms_omti(now_omti),4,2)+'s UT'
        PRINT,'Time limits for Keogram                                  '+ $
               STRING(omti_keo_stt_org,FORMAT='(I4.4)')+' UT to '+ $
	       STRING(omti_keo_end_org,FORMAT='(I4.4)')+' UT'

; Print info. for each image
        IF KEYWORD_SET(image_info) THEN BEGIN
        	PRINT,'---------------------------------------------------------------------------'
		omti_tcnt=720

; Check time info.
		time_info=STRARR(omti_tcnt)
		FOR k=0,omti_tcnt-1 DO BEGIN
			IF avl_omti(k) EQ 0 THEN BEGIN
				time_info(k)='no image'
			ENDIF ELSE BEGIN
        			time_info(k)=STRMID(hms_omti(k),0,4)+' '+STRMID(hms_omti(k),4,2)+'s'
			ENDELSE
		ENDFOR
		num_row=4
                off=FIX(omti_tcnt/num_row)
                dif=num_row-omti_tcnt+off*num_row
                IF dif EQ num_row THEN BEGIN
                        dif=0 & off=off-1
                ENDIF
                FOR i=0,off-dif DO BEGIN
                        PRINT,'Img '+STRING(i,FORMAT='(I3.3)')+' ('+time_info(i)+')' $
                        +' Img '+STRING(i+off*1+1,FORMAT='(I3.3)')+' ('+ $
                                time_info(i+off*1+1)+')' $
                        +' Img '+STRING(i+off*2+2,FORMAT='(I3.3)')+' ('+ $
                                time_info(i+off*2+2)+')' $
                        +' Img '+STRING(i+off*3+3,FORMAT='(I3.3)')+' ('+ $
                                time_info(i+off*3+3)+')'
                ENDFOR
        ENDIF
        PRINT,'---------------------------------------------------------------------------'

        END

;----------------------------------------------------------------------------------------------------------
; NAME:
;
;	FILE_OMTI_FOV
;
; PURPOSE:
;
;	Capture field-of-view of the ASI.
;

	PRO file_omti_fov,make_map=make_map,alt=alt

	COMMON data_omti
	COMMON pref_omti

; Map file base
	IF NOT KEYWORD_SET(alt) THEN alt=250
	alt_omti=alt
	fname='coords_'+STRING(alt,FORMAT='(I3.3)')

; Read geographic coords.
	OPENR,fov_lun,omti_cord_dir+'/'+fname+'.dat',/GET_LUN
	READF,fov_lun,glo_omti
	READF,fov_lun,gla_omti
	READF,fov_lun,zan_omti
	FREE_LUN,fov_lun

; Read geomagnetic coords.
	OPENR,fov_lun,omti_cord_dir+'/'+fname+'_mag.dat',/GET_LUN
	READF,fov_lun,mlo_omti
	READF,fov_lun,mla_omti
	FREE_LUN,fov_lun

	END

;----------------------------------------------------------------------------------------------------------
; NAME:
;
;	PLOT_OMTI_PANEL
;
; PURPOSE:
;
;	Plot OMTI data in all-sky image (as you can see from the observatory!).
;
; EXAMPLE:
;
;	Go > plot_omti_panel,1,2,0,0
;	Go > plot_omti_panel,1,2,0,0,scan=100
;	Go > plot_omti_panel,1,2,0,0,max_val=1000
;

	PRO plot_omti_panel,xmaps,ymaps,xmap,ymap,position=position,scan=scan, $
		max_val=max_val,min_val=min_val

	COMMON data_omti
	COMMON pref_omti

	COMMON prefs
	COMMON colour_info
	update_colour_info

; Define position
        IF N_PARAMS() NE 4 THEN BEGIN
                xmaps=1 & ymaps=1 & xmap=0 & ymap=0
        ENDIF

; Define panel position
	set_min_charsize,0.16
        define_panel,xmaps,ymaps,xmap,ymap,/bar,/square
        IF KEYWORD_SET(position) THEN !P.POSITION=position

; Resize charsize
	IF !D.NAME EQ 'PS' THEN !P.CHARSIZE=MAX([1.2*!P.CHARSIZE,0.32])
	IF !D.NAME NE 'PS' THEN !P.CHARSIZE=MIN([2.5*!P.CHARSIZE,1.50])
	IF KEYWORD_SET(force_char) THEN !P.CHARSIZE=force_char

; Font (to postscript font)
        IF !D.NAME EQ 'PS' AND ps_font EQ 1 THEN !P.FONT=0 ELSE !P.FONT=-1

; Scan identify
	IF NOT KEYWORD_SET(scan) THEN scan=now_omti

; If specified scan is out of time range
	IF scan GT num_omti THEN BEGIN
		PRINT,'WARNING: Scan out of range, plot is not created!'
		RETURN
	ENDIF
	IF avl_omti(scan) EQ 0 THEN BEGIN
		PRINT,'WARNING: Scan out of range, plot is not created!'
		RETURN
	ENDIF

; Max value setmping
	IF NOT KEYWORD_SET(min_val) THEN min_val=lev_omti(0)
	IF NOT KEYWORD_SET(max_val) THEN max_val=lev_omti(1)

; X-Y ranging
	xrange=xra_omti
	yrange=yra_omti

; Plot frame
	no_tick=MAKE_ARRAY(11,/STRING,VALUE=' ')
	IF KEYWORD_SET(no_frame) THEN col=background ELSE col=foreground
	PLOT,[0],[0],XRANGE=xrange,YRANGE=yrange,/XSTYLE,/YSTYLE,XTICKLEN=-0.01, $
		YTICKLEN=-0.01,XTICKS=10,YTICKS=10,XTICKNAME=no_tick,YTICKNAME=no_tick,COL=col

; Change colour table
	IF avl_omti(scan) EQ 0 THEN cut_col_tab ELSE loadct,col_omti,/silent

; Plot the data in contour form
	contour_lines=30
	mat_cont=INDGEN(256)+0.5
	levels_set=(max_val-min_val)*FINDGEN(contour_lines)/FLOAT(contour_lines)+min_val
	colour_set=!D.TABLE_SIZE*INDGEN(contour_lines)/contour_lines
	tmp_plot=INTARR(256,256)
	FOR xx=0,255 DO BEGIN
		FOR yy=0,255 DO BEGIN
			tmp_plot(xx,yy)=img_omti(xx,yy,scan)
			IF tmp_plot(xx,yy) GT max_val THEN tmp_plot(xx,yy)=max_val
			IF tmp_plot(xx,yy) LT min_val THEN tmp_plot(xx,yy)=min_val
		ENDFOR
	ENDFOR
	CONTOUR,tmp_plot,mat_cont,mat_cont,/OVERPLOT,/FILL, $
		LEVELS=levels_set,C_COLORS=colour_set

; Change colour table back to original
	IF avl_omti(scan) EQ 0 THEN loadct,col_omti,/silent

; Mask no data area
	cut_col_tab
	FOR i=0,180 DO BEGIN
		x1=xce_omti+111*COS(2*(i-1)*!PI/180.0)
		x2=xce_omti+111*COS(2*(i+1)*!PI/180.0)
		x3=xce_omti+180*COS(2*(i+1)*!PI/180.0)
		x4=xce_omti+180*COS(2*(i-1)*!PI/180.0)
		y1=yce_omti+111*SIN(2*(i-1)*!PI/180.0)
		y2=yce_omti+111*SIN(2*(i+1)*!PI/180.0)
		y3=yce_omti+180*SIN(2*(i+1)*!PI/180.0)
		y4=yce_omti+180*SIN(2*(i-1)*!PI/180.0)
		IF KEYWORD_SET(no_frame) THEN col=background ELSE col=253
		POLYFILL,[x1,x2,x3,x4],[y1,y2,y3,y4],COL=col,NOCLIP=0
	ENDFOR
	loadct,col_omti,/silent

; Overplot frame
	PLOT,[0],[0],XRANGE=xra_omti,YRANGE=yra_omti,/XSTYLE,/YSTYLE,XTICKLEN=-0.01, $
		YTICKLEN=-0.01,XTICKS=10,YTICKS=10,XTICKNAME=no_tick,YTICKNAME=no_tick

; Plot direction
	IF NOT KEYWORD_SET(no_frame) THEN BEGIN
		XYOUTS,xra_omti(0)-7, yce_omti,'E',SIZE=!P.CHARSIZE,ALI=0.5,ORI=90
		XYOUTS,xra_omti(1)+7, yce_omti,'W',SIZE=!P.CHARSIZE,ALI=0.5,ORI=270
		XYOUTS,xce_omti,yra_omti(0)-13,'S',SIZE=!P.CHARSIZE,ALI=0.5
		XYOUTS,xce_omti, yra_omti(1)+5,'N',SIZE=!P.CHARSIZE,ALI=0.5
	ENDIF

; Plot lines for keogram
	IF NOT KEYWORD_SET(no_frame) THEN BEGIN
		OPLOT,[xce_omti,xce_omti],yra_omti,COLOR=background,LINE=2,THICK=2
		OPLOT,xra_omti,[yce_omti,yce_omti],COLOR=background,LINE=2,THICK=2
		OPLOT,xra_omti,[yra_omti(0),yra_omti(1)],COLOR=background,LINE=2,THICK=2
		OPLOT,xra_omti,[yra_omti(1),yra_omti(0)],COLOR=background,LINE=2,THICK=2
	ENDIF

; Plot time info.
	time_info=STRMID(hms_omti(scan),0,4)+' '+STRMID(hms_omti(scan),4,2)+'s UT'
	XYOUTS,xra_omti(1),yra_omti(1)+5,time_info,SIZE=1.1*!P.CHARSIZE,ALI=1.0

; Plot filter info.
	IF flt_omti EQ 'C61' THEN flt_info='OI 557.7 nm'
	IF flt_omti EQ 'C62' THEN flt_info='OI 630.0 nm'
	IF flt_omti EQ 'C66' THEN flt_info='Sodium Line'
	XYOUTS,xra_omti(0),yra_omti(1)+5,flt_info,SIZE=1.1*!P.CHARSIZE,ALI=0.0

; Print Message for check
        PRINT,'Scan '+STRING(now_omti,FORMAT='(I4.4)')+' (000-'+ $
		STRING(num_omti-1,FORMAT='(I4.4)')+') '+hms_omti(now_omti)

; Overplot frame
	PLOT,[0],[0],XRANGE=xrange,YRANGE=yrange,/XSTYLE,/YSTYLE,XTICKLEN=-0.01, $
		YTICKLEN=-0.01,XTICKS=10,YTICKS=10,XTICKNAME=no_tick,YTICKNAME=no_tick,COL=foreground

; Initialize min charsize
	set_min_charsize,0.32
	
	END

;----------------------------------------------------------------------------------------------------------
; NAME:
;
;	PLOT_OMTI
;
; PURPOSE:
;
;	Plot the data in all-sky image. Multiple panels can be plotted in one window.
;
; EXAMPLE:
;
;	Go > plot_omti,2,3
;	Go > plot_omti,3,4,max_val=1200
;

	PRO plot_omti,xmaps,ymaps,max_val=max_val,min_val=min_val

	COMMON data_omti
	COMMON pref_omti

; Charset
	!P.CHARSIZE=1.0

; Page clearance
	clear_page

; Panel positioning
        IF N_PARAMS() NE 2 THEN BEGIN
                xmaps=1 & ymaps=1
        ENDIF

; Max value setmping
	IF NOT KEYWORD_SET(max_val) THEN max_val=0
	IF NOT KEYWORD_SET(min_val) THEN min_val=0

; Plot title and colour bar
	plot_omti_title
	loadct,col_omti,/silent

	IF !D.NAME EQ 'PS' THEN BEGIN
		plot_omti_colour_bar,1,0,max_val=max_val,min_val=min_val,force_char=0.5,leg_pos=1.2
	ENDIF ELSE BEGIN
		plot_omti_colour_bar,1,0,max_val=max_val,min_val=min_val,force_char=1.0
	ENDELSE

; Define panels and plot the data
        FOR i=0,xmaps*ymaps-1 DO BEGIN

; End of file identification
                IF now_omti GT num_omti-1 THEN BEGIN
                        PRINT,'End of file reached'
                        RETURN
                ENDIF

; Panel management
                y_div=i/xmaps & x_div=i-xmaps*(i/xmaps)

; Plot the panel
		IF avl_omti(now_omti) NE 0 THEN $
                	plot_omti_panel,xmaps,ymaps,x_div,y_div,max_val=max_val,min_val=min_val

; Progress the scan
                now_omti++
        ENDFOR

	END

;----------------------------------------------------------------------------------------------------------
; NAME:
;
;	PLOT_OMTI_MAP_PANEL
;
; PURPOSE:
;
;	Plot OMTI data in latitude vs longitude format.	Geographic and geomagnetic coords are supported.
;
; EXAMPLE:
;
;	Go > plot_omti_map_panel,2,2,0,0
;	Go > plot_omti_map_panel,3,3,0,0,max_val=2400
;

	PRO plot_omti_map_panel,xmaps,ymaps,xmap,ymap,position=position,scan=scan, $
		max_val=max_val,min_val=min_val,no_x=no_x,no_y=no_y

	COMMON data_omti
	COMMON pref_omti

	COMMON prefs
	COMMON colour_info
	update_colour_info

; Define position
        IF N_PARAMS() NE 4 THEN BEGIN
                xmaps=1 & ymaps=1 & xmap=0 & ymap=0
        ENDIF

; Define panel position
	set_min_charsize,0.16
        define_panel,xmaps,ymaps,xmap,ymap,/bar,/square
        IF KEYWORD_SET(position) THEN !P.POSITION=position

; Resize charsize
	IF !D.NAME EQ 'PS' THEN !P.CHARSIZE=MAX([1.2*!P.CHARSIZE,0.32])
	IF !D.NAME NE 'PS' THEN !P.CHARSIZE=MIN([2.5*!P.CHARSIZE,1.50])

; Set up contour line thickness
	IF !D.NAME EQ 'PS' THEN c_thick=3
	IF !D.NAME NE 'PS' THEN c_thick=1

; Font (to postscript font)
        IF !D.NAME EQ 'PS' AND ps_font EQ 1 THEN !P.FONT=0 ELSE !P.FONT=-1

; Scan identify
	IF NOT KEYWORD_SET(scan) THEN scan=now_omti

; If specified scan is out of time range
	IF scan GT num_omti THEN BEGIN
		PRINT,'WARNING: Scan out of range, plot is not created!'
		RETURN
	ENDIF
	IF avl_omti(scan) EQ 0 THEN BEGIN
		PRINT,'WARNING: Scan out of range, plot is not created!'
		RETURN
	ENDIF

; Max value setmping
	IF NOT KEYWORD_SET(min_val) THEN min_val=lev_omti(0)
	IF NOT KEYWORD_SET(max_val) THEN max_val=lev_omti(1)

; Range in x and y
	IF crd_omti EQ 0 THEN BEGIN
		xrange=[250,280] & yrange=[66,84]
		lat_omti=gla_omti & lon_omti=glo_omti
		xtit='Geographic Longitude' & ytit='Geographic Latitude'
		xtick=['250','255','260','265','270','275','280']
		ytick=['66','69','72','75','78','81','84']
		xticks=6 & yticks=6
	ENDIF
	IF crd_omti EQ 1 THEN BEGIN
		xrange=[-70,0] & yrange=[76,90]
		lat_omti=mla_omti & lon_omti=mlo_omti
		xtit='Magnetic Longitude' & ytit='Magnetic Latitude'
		xtick=['-70','-60','-50','-40','-30','-20','-10','0']
		ytick=['76','78','80','82','84','86','88','90']
		xticks=7 & yticks=7
	ENDIF

; Plot frame
	no_tick=MAKE_ARRAY(8,/STRING,VALUE=' ')
	IF KEYWORD_SET(no_x) THEN BEGIN
		xtit=''
	ENDIF
	IF KEYWORD_SET(no_y) THEN BEGIN
		ytit=''
	ENDIF
	PLOT,[0],[0],XRANGE=xrange,YRANGE=yrange,/XSTYLE,/YSTYLE,XTICKLEN=-0.01, $
		YTICKLEN=-0.01,XTICKS=xticks,YTICKS=xticks,XMINOR=5,YMINOR=3, $
		XTICKNAME=xtick,YTICKNAME=ytick,XTITLE=xtit,YTITLE=ytit

; Change colour table
	IF avl_omti(scan) EQ 0 THEN cut_col_tab ELSE loadct,col_omti,/silent

; Plot data in contour form
	new_plot=INTARR(256,256)
	FOR xx=0,255 DO BEGIN
		FOR yy=0,255 DO BEGIN
			new_plot(xx,yy)=img_omti(xx,yy,scan)
			IF new_plot(xx,yy) GT max_val THEN new_plot(xx,yy)=max_val
			IF new_plot(xx,yy) LT min_val THEN new_plot(xx,yy)=min_val
			IF lat_omti(xx,yy) EQ -999.0 AND lon_omti(xx,yy) EQ -999.9 THEN BEGIN
				new_plot(xx,yy)=-30001
			ENDIF
		ENDFOR
	ENDFOR
	contour_lines=30
	levels_set=(max_val-min_val)*FINDGEN(contour_lines)/FLOAT(contour_lines)+min_val
	colour_set=!D.TABLE_SIZE*INDGEN(contour_lines)/contour_lines
	CONTOUR,new_plot,lon_omti,lat_omti,/OVERPLOT,/FILL,LEVELS=levels_set, $
		C_COLORS=colour_set,MIN_VALUE=-30000,/IRREGULAR,NOCLIP=0

; Change colour table back to original
	IF avl_omti(scan) EQ 0 THEN loadct,col_omti,/silent

; Overplot frame
	PLOT,[0],[0],XRANGE=xrange,YRANGE=yrange,/XSTYLE,/YSTYLE,XTICKLEN=-0.01, $
		YTICKLEN=-0.01,XTICKS=xticks,YTICKS=yticks,XTICKNAME=no_tick,YTICKNAME=no_tick

; Plot time info.
	pos=!P.POSITION
	v_offset=(!P.POSITION(3)-!P.POSITION(1))*0.02
	time_info=STRMID(hms_omti(scan),0,4)+' '+STRMID(hms_omti(scan),4,2)+'s UT'
	XYOUTS,pos(2),pos(3)+v_offset,time_info,SIZE=1.1*!P.CHARSIZE,ALI=1.0,/NORMAL

; Plot filter info.
	IF flt_omti EQ 'C61' THEN flt_info='OI 557.7 nm'
	IF flt_omti EQ 'C62' THEN flt_info='OI 630.0 nm'
	IF flt_omti EQ 'C66' THEN flt_info='Sodium Line'
	XYOUTS,pos(0),pos(3)+v_offset,flt_info,SIZE=1.1*!P.CHARSIZE,ALI=0.0,/NORMAL

; Print Message for check
        PRINT,'Scan '+STRING(now_omti,FORMAT='(I4.4)')+' (000-'+ $
		STRING(num_omti-1,FORMAT='(I4.4)')+') '+hms_omti(now_omti)

; Initialize min charsize
	set_min_charsize,0.32
	
	END

;----------------------------------------------------------------------------------------------------------
; NAME:
;
;	PLOT_OMTI_MAP
;
; PURPOSE:
;
;	Plot OMTI data in latitude vs longitude format.	Multiple panels can be plotted in one window.
;
; EXAMPLE:
;
;	Go > plot_omti_map,1,2,0,0
;	Go > plot_omti_map,2,3,0,0,max_val=800
;

        PRO plot_omti_map,xmaps,ymaps,max_val=max_val,min_val=min_val

        COMMON data_omti
        COMMON pref_omti

; Charset
        !P.CHARSIZE=1.0

; Page clearance
        clear_page

; Panel positioning
        IF N_PARAMS() NE 2 THEN BEGIN
                xmaps=1 & ymaps=1
        ENDIF

; Max value setmping
        IF NOT KEYWORD_SET(max_val) THEN max_val=0
        IF NOT KEYWORD_SET(min_val) THEN min_val=0

; Plot title and colour bar
        plot_omti_title
        loadct,col_omti,/silent

        IF !D.NAME EQ 'PS' THEN BEGIN
                plot_omti_colour_bar,1,0,max_val=max_val,min_val=min_val,force_char=0.6,leg_pos=1.2
        ENDIF ELSE BEGIN
                plot_omti_colour_bar,1,0,max_val=max_val,min_val=min_val,force_char=1.0
        ENDELSE

; Define panels and plot the data
        FOR i=0,xmaps*ymaps-1 DO BEGIN

; End of file identification
                IF now_omti GT num_omti-1 THEN BEGIN
                        PRINT,'End of file reached'
                        RETURN
                ENDIF

; Panel management
                y_div=i/xmaps & x_div=i-xmaps*(i/xmaps)

; Plot the panel
		no_x=1 & no_y=1
		IF y_div EQ ymaps-1 THEN no_x=0
		IF x_div EQ       0 THEN no_y=0
                IF avl_omti(now_omti) NE 0 THEN BEGIN
                        plot_omti_map_panel,xmaps,ymaps,x_div,y_div,max_val=max_val, $
				min_val=min_val,no_x=no_x,no_y=no_y
                ENDIF

; Progress the scan
                now_omti++
        ENDFOR

	END

;----------------------------------------------------------------------------------------------------------
; NAME:
;
;	PLOT_OMTI_POLAR_PANEL
;
; PURPOSE:
;
;	Plot OMTI data in magnetic latitude and MLT coordinate system (AACGM).
;
; EXAMPLE:
;
;	Go > plot_omti_polar_panel,1,2,0,0
;	Go > plot_omti_polar_panel,1,2,0,0,/clip
;

        PRO plot_omti_polar_panel,xmaps,ymaps,xmap,ymap,position=position,scan=scan,clip=clip, $
                max_val=max_val,min_val=min_val,xrange=xrange,yrange=yrange,ticklen=ticklen

        COMMON data_omti
        COMMON pref_omti

	COMMON data
        COMMON prefs
        COMMON colour_info

; Update colour info on Go
        update_colour_info

; Define position
        IF N_PARAMS() NE 4 THEN BEGIN
                xmaps=1 & ymaps=1 & xmap=0 & ymap=0
        ENDIF

; Set coordinate system to mag
	lat_omti=mla_omti & lon_omti=mlo_omti+360.0

; Clip or not
	IF NOT KEYWORD_SET(clip) THEN clip=0

; Define panel position
        set_min_charsize,0.16
        define_panel,xmaps,ymaps,xmap,ymap,/bar,/square
        IF KEYWORD_SET(position) THEN !P.POSITION=position

; Resize charsize
        IF !D.NAME EQ 'PS' THEN !P.CHARSIZE=MAX([1.2*!P.CHARSIZE,0.32])
        IF !D.NAME NE 'PS' THEN !P.CHARSIZE=MIN([2.5*!P.CHARSIZE,1.50])

; Set up contour line thickness
        IF !D.NAME EQ 'PS' THEN c_thick=3
        IF !D.NAME NE 'PS' THEN c_thick=1

; Font (to postscript font)
        IF !D.NAME EQ 'PS' AND ps_font EQ 1 THEN !P.FONT=0 ELSE !P.FONT=-1

; Scan identify
        IF NOT KEYWORD_SET(scan) THEN scan=now_omti

; If specified scan is out of time range
        IF scan GT num_omti THEN BEGIN
                PRINT,'WARNING: Scan out of range, plot is not created!'
                RETURN
        ENDIF
        IF avl_omti(scan) EQ 0 THEN BEGIN
                PRINT,'WARNING: Scan out of range, plot is not created!'
                RETURN
        ENDIF

; Max value setmping
        IF NOT KEYWORD_SET(min_val) THEN min_val=lev_omti(0)
        IF NOT KEYWORD_SET(max_val) THEN max_val=lev_omti(1)

; Date identification
        yy=FIX(STRMID(day_omti,0,4))
        mm=FIX(STRMID(day_omti,4,2))
        dd=FIX(STRMID(day_omti,6,2))
        omti_jday=JULDAY(mm,dd,yy)-JULDAY(1,0,yy)
	year_secs=86400L*omti_jday+tsc_omti(scan)

; Plot panel
	IF NOT KEYWORD_SET(xrange) OR NOT KEYWORD_SET(yrange) THEN BEGIN
		xrange=[-31,31] & yrange=[-31,31]
		IF KEYWORD_SET(clip) THEN BEGIN
			pos=cnvcoord(lat_rsb,lon_rsb,1)
			mla_rsb=pos(0) & mlo_rsb=pos(1)
			x1= abs(90-mla_rsb)*SIN(mlt(yy,year_secs,mlo_rsb)*!PI/12)
                	y1=-abs(90-mla_rsb)*COS(mlt(yy,year_secs,mlo_rsb)*!Pi/12)
			IF alt_omti GE 200 THEN xyrange=alt_omti/10.0 ELSE xyrange=alt_omti/8.0
                	xrange=[x1-xyrange*0.5,x1+xyrange*0.5]
                	yrange=[y1-xyrange*0.5,y1+xyrange*0.5]
		ENDIF
	ENDIF

; File radar data if not stored
	IF NOT KEYWORD_SET(ticklen) THEN ticklen=!P.TICKLEN
	plot_polar_frame,position=!P.POSITION,xrange=xrange,yrange=yrange,ticklen=ticklen

; Change colour table
        IF avl_omti(scan) EQ 0 THEN cut_col_tab ELSE loadct,col_omti,/silent

; Plot data in contour form
        tmp_bin=WHERE(gla_omti NE -999.0 AND gla_omti NE -999.0 AND zan_omti LE 80.0,no_tmp_bin)
	cont=MAKE_ARRAY(4,no_tmp_bin,/FLOAT,VALUE=0.0)
	num_cnt=0L
        FOR xxx=0,255 DO BEGIN
                FOR yyy=0,255 DO BEGIN
                        IF gla_omti(xxx,yyy) NE -999.0 AND gla_omti(xxx,yyy) NE -999.0 AND $
				zan_omti(xxx,yyy) LE 80.0 THEN BEGIN

				cont(2,num_cnt)=FLOAT(img_omti(xxx,yyy,scan))
				cont(3,num_cnt)=zan_omti(xxx,yyy)
                        	IF cont(2,num_cnt) GT max_val THEN cont(2,num_cnt)=max_val
                        	IF cont(2,num_cnt) LT min_val THEN cont(2,num_cnt)=min_val
				cont(0,num_cnt)= ABS(90-lat_omti(xxx,yyy)) $
					*SIN(mlt(yy,year_secs,lon_omti(xxx,yyy))*!PI/12)
				cont(1,num_cnt)=-ABS(90-lat_omti(xxx,yyy)) $
					*COS(mlt(yy,year_secs,lon_omti(xxx,yyy))*!PI/12)
				num_cnt++
			ENDIF
		ENDFOR
	ENDFOR
        contour_lines=30
        levels_set=(max_val-min_val)*FINDGEN(contour_lines)/FLOAT(contour_lines)+min_val
        colour_set=!D.TABLE_SIZE*INDGEN(contour_lines)/contour_lines
        CONTOUR,cont(2,*),cont(0,*),cont(1,*),/OVERPLOT,/FILL,LEVELS=levels_set, $
                C_COLORS=colour_set,/IRREGULAR,NOCLIP=0
 
; Change colour table back to original
        IF avl_omti(scan) EQ 0 THEN loadct,col_omti,/silent

; Overplot frame
	plot_polar_frame,position=!P.POSITION,xrange=xrange,yrange=yrange,ticklen=ticklen

; Plot MLT
	IF xrange(0)*xrange(1) LE 0 AND yrange(0) LE 0 THEN BEGIN
		y_plot=yrange(0)-0.12*(yrange(1)-yrange(0))
		XYOUTS,0,y_plot,'24MLT',ALI=0.5,/DATA
	ENDIF
	IF NOT KEYWORD_SET(no_dusk) THEN BEGIN
		IF xrange(0) LT -10 AND yrange(1) GE 0 THEN BEGIN
			x_plot=xrange(0)-0.05*(xrange(1)-xrange(0))
			XYOUTS,x_plot,0,'18MLT',ALI=0.5,/DATA,ORI=90
		ENDIF
	ENDIF
	IF NOT KEYWORD_SET(no_dawn) THEN BEGIN
		IF xrange(1) GT +10 AND yrange(1) GE 0 THEN BEGIN
			IF KEYWORD_SET(clip) THEN BEGIN
				x_plot=xrange(1)+0.1*(xrange(1)-xrange(0))
			ENDIF ELSE BEGIN
				x_plot=xrange(1)+0.05*(xrange(1)-xrange(0))
			ENDELSE
			XYOUTS,x_plot,0,'06MLT',ALI=0.5,/DATA,ORI=270
		ENDIF
	ENDIF

; Plot time info.
	time_info=STRMID(hms_omti(scan),0,4)+' '+STRMID(hms_omti(scan),4,2)+'s UT'
	IF KEYWORD_SET(no_time) THEN time_info=' '
        pos=!P.POSITION
        v_offset=(pos(3)-pos(1))*0.04
       	XYOUTS,pos(2),pos(3)+v_offset,time_info,SIZE=1.1*!P.CHARSIZE,ALI=1.0,/NORMAL

; Plot filter info.
	IF flt_omti EQ 'C61' THEN flt_info='OI 557.7 nm'
        IF flt_omti EQ 'C62' THEN flt_info='OI 630.0 nm'
        IF flt_omti EQ 'C66' THEN flt_info='Sodium Line'
        XYOUTS,pos(0),pos(3)+v_offset,flt_info,SIZE=1.1*!P.CHARSIZE,ALI=0.0,/NORMAL

; Print Message for check
        PRINT,'Scan '+STRING(now_omti,FORMAT='(I4.4)')+' (000-'+ $
                STRING(num_omti-1,FORMAT='(I4.4)')+') '+hms_omti(now_omti)

; Initialize min charsize
        set_min_charsize,0.32
        
        END

;----------------------------------------------------------------------------------------------------------
; NAME:
;
;       PLOT_OMTI_POLAR
;
; PURPOSE:
;
;	Plot OMTI data in magnetic latitude and MLT coordinate system (AACGM). Multiple panels can be
;	plotted in one window.
;
; EXAMPLE:
;
;	Go > plot_omti_polar,2,2
;	Go > plot_omti_polar,2,2,/clip
;

        PRO plot_omti_polar,xmaps,ymaps,max_val=max_val,min_val=min_val,clip=clip

        COMMON data_omti
        COMMON pref_omti

; Charset
        !P.CHARSIZE=1.0

; Page clearance
        clear_page

; Panel positioning
        IF N_PARAMS() NE 2 THEN BEGIN
                xmaps=1 & ymaps=1
        ENDIF

; Max value setmping
        IF NOT KEYWORD_SET(max_val) THEN max_val=0
        IF NOT KEYWORD_SET(min_val) THEN min_val=0

; Clip or not
        IF NOT KEYWORD_SET(clip) THEN clip=0

; Plot title and colour bar
        plot_omti_title
        loadct,col_omti,/silent

        IF !D.NAME EQ 'PS' THEN BEGIN
		IF KEYWORD_SET(pot) THEN BEGIN
                	plot_omti_colour_bar,2,0,max_val=max_val,min_val=min_val, $
				force_char=0.6,leg_pos=1.2
		ENDIF ELSE BEGIN
                	plot_omti_colour_bar,1,0,max_val=max_val,min_val=min_val, $
				force_char=0.6,leg_pos=1.2
		ENDELSE
        ENDIF ELSE BEGIN
		IF KEYWORD_SET(pot) THEN BEGIN
                	plot_omti_colour_bar,2,0,max_val=max_val,min_val=min_val,force_char=1.0
		ENDIF ELSE BEGIN
                	plot_omti_colour_bar,1,0,max_val=max_val,min_val=min_val,force_char=1.0
		ENDELSE
        ENDELSE

; Define panels and plot the data
        FOR i=0,xmaps*ymaps-1 DO BEGIN

; End of file identification
                IF now_omti GT num_omti-1 THEN BEGIN
                        PRINT,'End of file reached'
                        RETURN
                ENDIF

; Panel management
                y_div=i/xmaps & x_div=i-xmaps*(i/xmaps)

; Plot the panel
                IF avl_omti(now_omti) NE 0 THEN BEGIN
                        plot_omti_polar_panel,xmaps,ymaps,x_div,y_div,max_val=max_val, $
                                min_val=min_val,clip=clip
                ENDIF

; Progress the scan
                now_omti++
        ENDFOR

        END

;----------------------------------------------------------------------------------------------------------
; NAME:
;
;	OVERLAY_OMTI_POLAR_DATA
;
; PURPOSE:
;
;	Overlay OMTI data in magnetic latitude and MLT coordinate system (AACGM).
;
; EXAMPLE:
;
;	Go > overlay_omti_polar_data
;	Go > overlay_omti_polar_data,max_val=1500
;

        PRO overlay_omti_polar_data,scan=scan,max_val=max_val,min_val=min_val

        COMMON data_omti
        COMMON pref_omti

	COMMON data
        COMMON prefs
        COMMON colour_info

; Update colour info on Go
        update_colour_info

; Set coordinate system to mag
	lat_omti=mla_omti & lon_omti=mlo_omti+360.0

; Set up contour line thickness
        IF !D.NAME EQ 'PS' THEN c_thick=3
        IF !D.NAME NE 'PS' THEN c_thick=1

; Scan identify
        IF NOT KEYWORD_SET(scan) THEN scan=now_omti

; If specified scan is out of time range
        IF scan GT num_omti THEN BEGIN
                PRINT,'WARNING: Scan out of range, plot is not created!'
                RETURN
        ENDIF
        IF avl_omti(scan) EQ 0 THEN BEGIN
                PRINT,'WARNING: Scan out of range, plot is not created!'
                RETURN
        ENDIF

; Max value setmping
        IF NOT KEYWORD_SET(min_val) THEN min_val=lev_omti(0)
        IF NOT KEYWORD_SET(max_val) THEN max_val=lev_omti(1)

; Date identification
        yy=FIX(STRMID(day_omti,0,4))
        mm=FIX(STRMID(day_omti,4,2))
        dd=FIX(STRMID(day_omti,6,2))
        omti_jday=JULDAY(mm,dd,yy)-JULDAY(1,0,yy)
	year_secs=86400L*omti_jday+tsc_omti(scan)

; Change colour table
        IF avl_omti(scan) EQ 0 THEN cut_col_tab ELSE loadct,col_omti,/silent

; Plot data in contour form
        tmp_bin=WHERE(gla_omti NE -999.0 AND gla_omti NE -999.0 AND zan_omti LE 80.0,no_tmp_bin)
	cont=MAKE_ARRAY(4,no_tmp_bin,/FLOAT,VALUE=0.0)
	num_cnt=0L
        FOR xxx=0,255 DO BEGIN
                FOR yyy=0,255 DO BEGIN
                        IF gla_omti(xxx,yyy) NE -999.0 AND gla_omti(xxx,yyy) NE -999.0 AND $
				zan_omti(xxx,yyy) LE 80.0 THEN BEGIN

				cont(2,num_cnt)=FLOAT(img_omti(xxx,yyy,scan))
				cont(3,num_cnt)=zan_omti(xxx,yyy)
                        	IF cont(2,num_cnt) GT max_val THEN cont(2,num_cnt)=max_val
                        	IF cont(2,num_cnt) LT min_val THEN cont(2,num_cnt)=min_val
				cont(0,num_cnt)= ABS(90-lat_omti(xxx,yyy)) $
					*SIN(mlt(yy,year_secs,lon_omti(xxx,yyy))*!PI/12)
				cont(1,num_cnt)=-ABS(90-lat_omti(xxx,yyy)) $
					*COS(mlt(yy,year_secs,lon_omti(xxx,yyy))*!PI/12)
				num_cnt++
			ENDIF
		ENDFOR
	ENDFOR
        contour_lines=30
        levels_set=(max_val-min_val)*FINDGEN(contour_lines)/FLOAT(contour_lines)+min_val
        colour_set=!D.TABLE_SIZE*INDGEN(contour_lines)/contour_lines
        CONTOUR,cont(2,*),cont(0,*),cont(1,*),/OVERPLOT,/FILL,LEVELS=levels_set, $
                C_COLORS=colour_set,/IRREGULAR,NOCLIP=0
 
; Change colour table back to original
        IF avl_omti(scan) EQ 0 THEN loadct,col_omti,/silent

        END

;----------------------------------------------------------------------------------------------------------
; NAME:
;
;	PLOT_OMTI_KEO_PANEL
;

        PRO plot_omti_keo_panel,xmaps,ymaps,xmap,ymap,position=position,keo=keo, $
		max_val=max_val,min_val=min_val,no_x=no_x,no_y=no_y,ticklen=ticklen

        COMMON data_omti
	COMMON pref_omti

	COMMON prefs

; Colour info.
        COMMON colour_info
        update_colour_info

; Position management
        IF N_PARAMS() NE 4 THEN BEGIN
                xmaps=1 & ymaps=1 & xmap=0 & ymap=0
        ENDIF

; Define panel position
        set_min_charsize,0.16
        define_panel,xmaps,ymaps,xmap,ymap,/bar
        IF KEYWORD_SET(position) THEN !P.POSITION=position

; Resize charsize
        IF !D.NAME EQ 'PS' THEN !P.CHARSIZE=MAX([1.0*!P.CHARSIZE,0.60])
        IF !D.NAME NE 'PS' THEN !P.CHARSIZE=MIN([3.5*!P.CHARSIZE,1.50])

; Font (to postscript font)
        IF !D.NAME EQ 'PS' AND ps_font EQ 1 THEN !P.FONT=0 ELSE !P.FONT=-1

; Max value setmping
        IF NOT KEYWORD_SET(min_val) THEN min_val=lev_omti(0)
        IF NOT KEYWORD_SET(max_val) THEN max_val=lev_omti(1)

; Which line is plotted?
	IF NOT KEYWORD_SET(keo) THEN keo=0
	yrange=[0,222]
	IF keo EQ 0 THEN ytit='!Ukeogram!N!CSE  to  NW'
	IF keo EQ 1 THEN ytit='!Ukeogram!N!CSW  to  NE'
	IF keo EQ 2 THEN ytit='!Ukeogram!N!CSouth to North'
	IF keo EQ 3 THEN ytit='!Ukeogram!N!CEast to West'

; Determine time spacing and UT axis
        ts=omti_keo_stt/3600.0 & te=omti_keo_end/3600.0
        time_axis_intervals,ts,te,minor_ticks,where_tick,no_of_ticks

        IF KEYWORD_SET(no_x) THEN BEGIN
                xtit=''
                xtickformat='no_ticks'
                IF no_x EQ 1 THEN ytickname=[' '] ELSE ytickname=['']
        ENDIF ELSE BEGIN
                xtit='UT'
                xtickformat='axis_time'
                ytickname=['']
        ENDELSE
        IF KEYWORD_SET(no_y) THEN BEGIN
                ytit=''
                ytickformat='no_ticks'
        ENDIF
        IF NOT KEYWORD_SET(ticklen) THEN ticklen=!P.TICKLEN
	IF keo EQ 0 THEN ytickname=['SE',' ','Z ',' ','NW']
	IF keo EQ 1 THEN ytickname=['SW',' ','Z ',' ','NE']
	IF keo EQ 2 THEN ytickname=['S ',' ','Z ',' ','N ']
	IF keo EQ 3 THEN ytickname=['E ',' ','Z ',' ','W ']

; Plot frame
	yticks=4 & yminor=4
        pos=!P.POSITION
	xy_ratio=3.0
        PLOT,[0],[0],XRANGE=[ts,te],/XSTYLE,YRANGE=yrange,/YSTYLE,XTITLE=xtit,YTITLE=ytit, $
                XTICKFORMAT=xtickformat,YTICKNAME=ytickname,YTICKS=yticks,COLOR=background, $
                YMINOR=yminor,XTICKLEN=xy_ratio*ticklen,YTICKLEN=ticklen,XTICKS=no_of_ticks, $
                XTICKV=where_tick,XMINOR=minor_ticks,YTICKFORMAT=ytickformat
	
; Plot the data in contour form
	loadct,col_omti,/silent
        contour_lines=30
        mat_x=(INDGEN(1440)+0.5)*120.0/3600.0
        mat_y=INDGEN(222)+0.5
        levels_set=(max_val-min_val)*FINDGEN(contour_lines)/FLOAT(contour_lines)+min_val
        colour_set=!D.TABLE_SIZE*INDGEN(contour_lines)/contour_lines
        tmp_plot=INTARR(1440,222)
        FOR xx=0,1439 DO BEGIN
               	FOR yy=0,221 DO BEGIN
                       	tmp_plot(xx,yy)=keo_omti(keo,yy,xx)
                       	IF tmp_plot(xx,yy) GT max_val THEN tmp_plot(xx,yy)=max_val
                       	IF tmp_plot(xx,yy) LT min_val THEN tmp_plot(xx,yy)=min_val
               	ENDFOR
        ENDFOR
        CONTOUR,tmp_plot,mat_x,mat_y,/OVERPLOT,/FILL,LEVELS=levels_set,C_COLORS=colour_set

; Mask data lack
	plot_scans=WHERE(avl_omti NE 1,no_plot_scans)
	cut_col_tab
	FOR i=0,no_plot_scans-1 DO BEGIN
		xt=120L*plot_scans(i)
               	x_poly=[xt-60,xt+180,xt+180,xt-60]/3600.0
               	y_poly=[yra_omti(0),yra_omti(0),yra_omti(1),yra_omti(1)]
               	POLYFILL,x_poly,y_poly,COLOR=253,NOCLIP=0
	ENDFOR
	IF keo LT 2 THEN BEGIN
		POLYFILL,[ts,te,te,ts],[0.0,0.0,33.0,33.0],COLOR=253,NOCLIP=0
		POLYFILL,[ts,te,te,ts],[190,190, 222, 222],COLOR=253,NOCLIP=0
	ENDIF
	loadct,col_omti,/silent

; Overplot frame
        PLOT,[0],[0],XRANGE=[ts,te],/XSTYLE,YRANGE=yrange,/YSTYLE,XTITLE=xtit,YTITLE=ytit, $
                XTICKFORMAT=xtickformat,YTICKNAME=ytickname,YTICKS=yticks, $
                YMINOR=yminor,XTICKLEN=xy_ratio*ticklen,YTICKLEN=ticklen,XTICKS=no_of_ticks, $
                XTICKV=where_tick,XMINOR=minor_ticks,YTICKFORMAT=ytickformat

; Plot colour bar
	IF NOT KEYWORD_SET(no_bar) THEN BEGIN
		plot_omti_colour_bar,ymaps,ymap,max_val=max_val,min_val=min_val, $
			force_char=MAX([0.8*!P.CHARSIZE,0.4]),leg_pos=1.0
	ENDIF

        END

;----------------------------------------------------------------------------------------------------------
; NAME:
;
;	PLOT_OMTI_KEO
;
; PURPOSE:
;
;	Plot asi data in keogram format for four directions.
;

	PRO plot_omti_keo,max_val=max_val,min_val=min_val

	IF NOT KEYWORD_SET(max_val) THEN max_val=0
	IF NOT KEYWORD_SET(min_val) THEN min_val=0

; Setting page
	set_format,/guppies,/portrait
	clear_page

; First write a title
	plot_omti_title
	
; Plot keograms in different cuts
	plot_omti_keo_panel,1,4,0,0,keo=0,max_val=max_val,min_val=min_val
	plot_omti_keo_panel,1,4,0,1,keo=1,max_val=max_val,min_val=min_val
	plot_omti_keo_panel,1,4,0,2,keo=2,max_val=max_val,min_val=min_val
	plot_omti_keo_panel,1,4,0,3,keo=3,max_val=max_val,min_val=min_val

	END

;----------------------------------------------------------------------------------------------------------
; NAME:
;
;	PLOT_OMTI_TITLE
;
; PURPOSE:
;
;	Plot title for asi plot
;

	PRO plot_omti_title,long=long,append=append,title=title

	COMMON data_omti
	COMMON pref_omti
	
	COMMON prefs

; Font (to postscript font)
        IF !D.NAME EQ 'PS' AND ps_font EQ 1 THEN !P.FONT=0 ELSE !P.FONT=-1

; Months
        mo_str=['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec']

; Plot main title
	IF KEYWORD_SET(long) THEN tita='OMTI ALL-SKY IMAGER ' ELSE tita='OMTI ASI '
	titb='PARAMETER PLOT'
	IF NOT KEYWORD_SET(title) THEN title='Imager No.6 at Resolute Bay: Absolute Intensity'
        plot_title,title,main_title=tita+titb,/nodate

; Find date
	IF KEYWORD_SET(append) THEN BEGIN
		day_bef=find_next_prev_day(day_omti,/prev)
		date_str_bef=STRMID(day_bef,6,2)+' '+mo_str(FIX(STRMID(day_bef,4,2))-1)+ $
			' '+STRMID(day_bef,0,4)
        	yyyy=FIX(STRMID(day_bef,0,4))
        	mm=FIX(STRMID(day_bef,4,2))
        	dd=FIX(STRMID(day_bef,6,2))
        	omti_jday_bef=JULDAY(mm,dd,yyyy)-JULDAY(1,0,yyyy)
	ENDIF
        date_str=STRMID(day_omti,6,2)+' '+mo_str(FIX(STRMID(day_omti,4,2))-1)+' '+STRMID(day_omti,0,4)
        yyyy=FIX(STRMID(day_omti,0,4))
        mm=FIX(STRMID(day_omti,4,2))
        dd=FIX(STRMID(day_omti,6,2))
        omti_jday=JULDAY(mm,dd,yyyy)-JULDAY(1,0,yyyy)

; Plot date
	IF KEYWORD_SET(append) THEN BEGIN
        	XYOUTS,0.87,0.89,'!5'+date_str+'!3!E ('+STRING(omti_jday,FORMAT='(I3.3)')+')!N', $
                	/NORMAL,ALIGNMENT=0.5,CHARSIZE=0.8
        	XYOUTS,0.87,0.92,'to',/NORMAL,ALIGNMENT=0.5,CHARSIZE=0.6
        	XYOUTS,0.87,0.945,'!5'+date_str_bef+ $
			'!3!E ('+STRING(omti_jday_bef,FORMAT='(I3.3)')+')!N',/NORMAL, $
			ALIGNMENT=0.5,CHARSIZE=0.8
		XYOUTS,0.87,0.86,'filter id: '+flt_omti+'!C(exposure 30sec)', $
			/NORMAL,ALIGNMENT=0.5,CHARSIZE=0.5,COLOR=foreground
	ENDIF ELSE BEGIN
        	XYOUTS,0.87,0.91,'!5'+date_str+'!3!E ('+STRING(omti_jday,FORMAT='(I3.3)')+')!N', $
                	/NORMAL,ALIGNMENT=0.5
		XYOUTS,0.87,0.88,'filter id: '+flt_omti+'!C(exposure 30sec)', $
			/NORMAL,ALIGNMENT=0.5,CHARSIZE=0.5,COLOR=foreground
	ENDELSE


	END

;----------------------------------------------------------------------------------------------------------
; NAME:
;
;	PLOT_OMTI_COLOUR_BAR
;
; PURPOSE:
;
;	Plot colour bar for asi.
;

	PRO plot_omti_colour_bar,ymaps,ymap,position=position,leg_pos=leg_pos, $
		force_char=force_char,max_val=max_val,min_val=min_val,legend=legend

	COMMON data_omti
	COMMON pref_omti

; Setting max value
	IF NOT KEYWORD_SET(min_val) THEN min_val=lev_omti(0)
	IF NOT KEYWORD_SET(max_val) THEN max_val=lev_omti(1)

; Legend
	IF NOT KEYWORD_SET(legend) THEN typ_omti_led='Rayleigh'

; Legend position setmping
	IF NOT KEYWORD_SET(leg_pos) THEN leg_pos=0

; Char size
	IF NOT KEYWORD_SET(force_char) THEN force_char=0

; Plot colour bar
	set_scale,min_val,max_val
	IF NOT KEYWORD_SET(position) THEN BEGIN
		plot_colour_bar,ymaps,ymap,legend=typ_omti_led,leg_pos=leg_pos,/no_gnd
	ENDIF ELSE BEGIN
		plot_colour_bar,position=position,legend=typ_omti_led,leg_pos=leg_pos,/no_gnd
	ENDELSE

    END

;----------------------------------------------------------------------------------------------------------
; NAME:
;
;	PLOT_POLAR_FARME
;
; PURPOSE:
;
;	Plot geomagnetic latitude and MLT frame. Position of the panel depend basically on !P.POSITION.
;
; EXAMPLE:
;
;	Go > plot_polar_frame
;

        PRO plot_polar_frame,xrange=xrange,yrange=yrange,ticklen=ticklen,position=position

        COMMON colour_info

; Update colour info
        update_colour_info

; Position
        IF KEYWORD_SET(position) THEN !P.POSITION=position

; Plotting parameters
        IF NOT KEYWORD_SET(ticklen) THEN ticklen=!P.TICKLEN
        ticks='no_ticks'

; Setup plotting field
        IF NOT KEYWORD_SET(xrange) OR NOT KEYWORD_SET(yrange) THEN BEGIN
                xrange=[-31,31]
                yrange=[-31,31]
        ENDIF

; Plot the panel and latitudinal grids
        FOR grid=10,60,10 DO BEGIN
                Lon=FINDGEN(101)*!pi/50
                coLat=REPLICATE(grid,101)
                IF grid EQ 10 THEN $
                        PLOT,coLat,Lon,/POLAR,LINE=1,COLOR=foreground, $
                                XSTYLE=1,YSTYLE=1,XTICKFORMAT=ticks,YTICKFORMAT=ticks, $
                                XRANGE=xrange,YRANGE=yrange,TICKLEN=ticklen
                OPLOT,coLat,Lon,/POLAR,LINE=1,COLOR=foreground
        ENDFOR

; Plot longitudinal grids
        FOR grid=0,23 DO BEGIN
                x1=[10*SIN(grid*!pi/12),80*SIN(grid*!pi/12)]
                y1=[10*COS(grid*!pi/12),80*COS(grid*!pi/12)]
                PLOTS,x1,y1,LINESTYLE=1,COLOR=foreground,NOCLIP=0
        ENDFOR

    END
